#include "db-migration.h"
#include "../exceptions/migration.h"
#include "db.h"
#include <sstream>
#include <string>

void DbMigrationUtil::migrate()
{
    std::string sql;
    sqlite3 *connection = DbUtil::connectDb();

    //Start transaction
    sqlite3_exec(connection, "BEGIN TRANSACTION;", NULL, NULL, NULL);

    try
    {
        createMigrationTable(connection);

        //Create queue_item table
        sql = "CREATE TABLE IF NOT EXISTS `queue_item` ("
              "`id` INTEGER NOT NULL,"
              "`queue_id` INTEGER DEFAULT NULL,"
              "`source` TEXT NOT NULL,"
              "`display_name` TEXT NOT NULL,"
              "`type` TEXT NOT NULL,"
              "`duration` INTEGER NOT NULL,"
              "`position` INTEGER NOT NULL,"
              "PRIMARY KEY (`id` AUTOINCREMENT),"
              "FOREIGN KEY (`queue_id`) REFERENCES `queue` (`id`) ON DELETE CASCADE"
              ");";
        executeMigration("create_queue_item_table", sql, connection);

        //Create queue table
        sql = "CREATE TABLE IF NOT EXISTS `queue` ("
              "`id` INTEGER NOT NULL,"
              "`guild_id` TEXT NOT NULL,"
              "`name` TEXT NOT NULL,"
              "`current_item_id` INTEGER DEFAULT NULL,"
              "PRIMARY KEY (`id` AUTOINCREMENT),"
              "FOREIGN KEY (`current_item_id`) REFERENCES `queue_item` (`id`) ON DELETE SET NULL"
              ");";
        executeMigration("create_queue_table", sql, connection);

        //Create search table
        sql = "CREATE TABLE IF NOT EXISTS `search` ("
              "`id` INTEGER NOT NULL,"
              "`term` TEXT NOT NULL,"
              "`result` TEXT NOT NULL,"
              "`created_at` TEXT NOT NULL,"
              "PRIMARY KEY (`id` AUTOINCREMENT)"
              ");";
        executeMigration("create_search_table", sql, connection);

        //Create shortcut table
        sql = "CREATE TABLE `shortcut` ("
              "`id` INTEGER NOT NULL,"
              "`guild_id` TEXT NOT NULL,"
              "`command` TEXT NOT NULL,"
              "`query` TEXT NOT NULL,"
              "`created_at` TEXT NOT NULL DEFAULT (datetime('now')),"
              "PRIMARY KEY (`id` AUTOINCREMENT)"
              ");";
        executeMigration("create_shortcut_table", sql, connection);

        //Commit transaction
        sqlite3_exec(connection, "COMMIT;", NULL, NULL, NULL);
    }
    catch (MigrationException &e)
    {
        //Rollback transaction, show error message and exit
        sqlite3_exec(connection, "ROLLBACK;", NULL, NULL, NULL);
        DbUtil::closeDb(connection);

        std::cerr << e.what() << std::endl;
        std::exit(10);
    }

    DbUtil::closeDb(connection);
}

void DbMigrationUtil::createMigrationTable(sqlite3 *connection)
{
    std::string sql = "CREATE TABLE IF NOT EXISTS `" + migrationTableName + "` (`id` INTEGER NOT NULL, `name` TEXT NOT NULL, `created_at` TEXT NOT NULL, PRIMARY KEY (`id` AUTOINCREMENT))";

    char *createTableError = NULL;
    int result = sqlite3_exec(connection, sql.c_str(), NULL, NULL, &createTableError);
    if (result != SQLITE_OK)
    {
        std::stringstream errorMessage;
        errorMessage << "Error creating migrations table: " << createTableError;
        throw MigrationException(errorMessage.str());
    }
}

void DbMigrationUtil::executeMigration(std::string migrationName, std::string sql, sqlite3 *connection)
{
    sqlite3_stmt *statement = NULL;

    //Check if migration has been executed
    std::string sqlMigrationExecuted = "SELECT name FROM `" + migrationTableName + "` WHERE name = ?;";
    sqlite3_prepare_v2(connection, sqlMigrationExecuted.c_str(), sqlMigrationExecuted.length(), &statement, nullptr);
    sqlite3_bind_text(statement, 1, migrationName.c_str(), migrationName.length(), SQLITE_TRANSIENT);

    if (sqlite3_step(statement) != SQLITE_ROW)
    {
        //Migration not executed
        char *migrationError = NULL;
        int result = sqlite3_exec(connection, sql.c_str(), NULL, NULL, &migrationError);
        if (result != SQLITE_OK)
        {
            std::stringstream errorMessage;
            errorMessage << "Error executing migration " << migrationName << ": " << migrationError;
            throw MigrationException(errorMessage.str());
        }

        //Set migration to executed
        sql = "INSERT INTO `" + migrationTableName + "` (`name`, `created_at`) VALUES  (?, datetime('now'))";
        sqlite3_prepare_v2(connection, sql.c_str(), sql.length(), &statement, nullptr);
        sqlite3_bind_text(statement, 1, migrationName.c_str(), migrationName.length(), SQLITE_TRANSIENT);

        if (sqlite3_step(statement) != SQLITE_DONE)
        {
            //Error marking migration as executed
            throw MigrationException("Error marking migration " + migrationName + " as executed");
        }
    }
    sqlite3_finalize(statement);
}
