#include <chrono>
#include "../entities/queue.h"
#include "../entities/queue-item.h"
#include "../utils/db.h"
#include "resume.h"

ResumeCommand::ResumeCommand(dpp::cluster &bot, PlayCommand &playCommand, std::map<dpp::snowflake, std::shared_future<void>> &playCommandFutures) : AbstractCommand(bot), playCommand(playCommand), playCommandFutures(playCommandFutures)
{
}

std::vector<std::string> ResumeCommand::getRegex()
{
    std::vector<std::string> regex;
    regex.push_back("^-resume$");
    regex.push_back("^-unpause$");
    return regex;
}

std::string ResumeCommand::getHelp()
{
    return "**-resume, -unpause:** Unpauses the playback if it is currently paused. If the bot is not in a voice channel it will join and play the last played song";
}

void ResumeCommand::execute(dpp::message_create_t event, std::string match1, std::string match2)
{
    //Get voice connection
    dpp::voiceconn *v = event.from->get_voice(event.msg.guild_id);

    if (v && v->voiceclient && v->voiceclient->is_ready())
    {

        if (v->voiceclient->is_paused())
        {
            //Audio paused --> unpause it
            v->voiceclient->pause_audio(false);
            success(event, "Playback unpaused");
        }
        else if (v->voiceclient->is_playing())
        {
            //Audio already playing --> show error message
            error(event, "Audio is not paused");
        }
    }
    else
    {
        //Currently not in voice channel --> join it
        dpp::guild *g = dpp::find_guild(event.msg.guild_id);
        if (!g->connect_member_voice(event.msg.author.id, false, true))
        {
            error(event, "You don't seem to be in a voice channel! :(");
            return;
        }

        //Wait for voice channel connection
        std::promise<bool> promise;
        std::shared_future<bool> future = promise.get_future();
        dpp::event_handle eventId = bot.on_voice_ready([&promise, event](const dpp::voice_ready_t &voiceEvent) {

            dpp::discord_voice_client *v = voiceEvent.voice_client;

            if(v && v->server_id == event.msg.guild_id && v->is_ready()) {
                promise.set_value(true);
            }
        });

        //Wait until bot joined voice channel
        std::future_status status;
        do
        {
            status = future.wait_for(std::chrono::milliseconds(500));
        } while (status != std::future_status::ready);

        //Remove event since it is not needed anymore
        bot.on_voice_ready.detach(eventId);

        v = event.from->get_voice(event.msg.guild_id);

        //Play last played song
        QueueEntity queue = DbUtil::getQueue(event.msg.guild_id);
        QueueItemEntity currentQueueItem = DbUtil::getCurrentQueueItem(queue.currentItemId);

        if (currentQueueItem.id == -1)
        {
            //Current queue item not found
            error(event, "Could not find current song");
            return;
        }

        //Join voice channel
        success(event, "Playback resumed");

        //Play last played song
        std::packaged_task<void()> packagedTask([this, v, event, currentQueueItem] {
            playCommand.play(v->voiceclient, event, currentQueueItem);
        });
        playCommandFutures[event.msg.guild_id] = packagedTask.get_future();

        std::thread thread(std::move(packagedTask));
        thread.detach();
    }
}
