#include <cmath>
#include <iomanip>
#include <iostream>
#include <vector>

#include "../entities/queue-item.h"
#include "../entities/queue.h"
#include "../utils/db.h"
#include "queue.h"

QueueCommand::QueueCommand(dpp::cluster &bot) : AbstractCommand(bot)
{

    bot.on_button_click([&bot, this](const dpp::button_click_t &event)
                        {

        QueueEntity queue = DbUtil::getQueue(event.command.guild_id);
        std::vector<QueueItemEntity> queueItems;

        //Variable used for previous and next buttons to calculate the correct offset
        std::string previousOffsetString = event.custom_id;
        replace(previousOffsetString, "first", "");
        replace(previousOffsetString, "last", "");
        replace(previousOffsetString, "current", "");
        replace(previousOffsetString, "previous", "");
        replace(previousOffsetString, "next", "");

        int previousOffset = std::stoi(previousOffsetString);
        int offset = -1;

        if(event.custom_id.rfind("first", 0) == 0)
        {
            //First page
            offset = 0;
        }
        else if(event.custom_id.rfind("last", 0) == 0)
        {
            //Last page --> show songs from smaller multiple of 10 onwards. Calculating with total count of queue items. E.g. total items = 291 --> 290
            offset = -2;
        }
        else if(event.custom_id.rfind("current", 0) == 0)
        {
            //Page with last played/currently playing song
            offset = -1;
        }
        else if(event.custom_id.rfind("previous", 0) == 0)
        {
            //Previous page
            offset = previousOffset - 10;
            if(offset < 0){
                //Prevent negative offset
                offset = 0;
            }
        }
        else if(event.custom_id.rfind("next", 0) == 0)
        {
            //Next page
            offset = previousOffset + 10;
        }

        dpp::message message = buildMessage(event, event.command.guild_id, event.command.channel_id, queue, offset);
        event.reply(dpp::ir_update_message, message); });
}

std::vector<std::string> QueueCommand::getRegex()
{
    std::vector<std::string> regex;
    regex.push_back("^-queue$");
    regex.push_back("^-q$");
    return regex;
}

std::string QueueCommand::getHelp()
{
    return "**-queue, -q:** Outputs the queue (10 items in total, depending on the current song)";
}

void QueueCommand::execute(dpp::message_create_t event, std::string match1, std::string match2)
{

    QueueEntity queue = DbUtil::getQueue(event.msg.guild_id);
    bot.message_create(buildMessage(event, event.msg.guild_id, event.msg.channel_id, queue));
}

dpp::message QueueCommand::buildMessage(dpp::event_dispatch_t event, dpp::snowflake guildId, dpp::snowflake channelId, QueueEntity queue, int offset)
{
    std::string message = "No songs in queue. Add some with -play";

    QueueItemEntity currentQueueItem = DbUtil::getCurrentQueueItem(queue.currentItemId);
    int countQueueItems = DbUtil::getCountQueueItems(queue.id);

    if (offset == -1)
    {
        if (currentQueueItem.id == -1)
        {
            //If current queue item is not set --> output last page
            offset = -2;
        }
        else
        {
            //Show songs from smaller multiple of 10 onwards calculating from the current item on
            offset = (currentQueueItem.rowNumber / 10) * 10;
        }
    }

    if (offset == -2)
    {
        //No current queue item --> fetch last page
        offset = (countQueueItems / 10) * 10;
    }

    std::vector<QueueItemEntity> queueItems = DbUtil::getQueueItems(queue.id, offset);
    if (queueItems.size() > 0)
    {
        message = "";
    }

    //Get voice connection
    bool isPlaying = true;
    dpp::voiceconn *v = event.from->get_voice(guildId);
    if (!v || !v->voiceclient || !v->voiceclient->is_ready() || !v->voiceclient->is_playing())
    {
        isPlaying = false;
    }

    for (QueueItemEntity queueItem : queueItems)
    {
        //Output current queue item
        std::string currentTextPreLine = "";
        std::string currentTextSufLine = "";
        if (queueItem.id == currentQueueItem.id)
        {
            if (isPlaying)
            {
                currentTextPreLine = "⬐ current";
                currentTextSufLine = "⬑ current";
            }
            else
            {
                currentTextPreLine = "⬐ last played";
                currentTextSufLine = "⬑ last played";
            }
        }

        if (!currentTextPreLine.empty())
        {
            message += currentTextPreLine + "\n";
        }

        message += std::to_string(queueItem.rowNumber) + ". \t" +
                   queueItem.getDisplayName() + "\t" +
                   intToTime(queueItem.duration) + " \n";

        if (!currentTextSufLine.empty())
        {
            message += currentTextSufLine + "\n";
        }
    }

    return dpp::message(channelId, createEmbed(0x00ff00, "", message)).add_component(getComponents(queueItems, offset, countQueueItems));
}

dpp::component QueueCommand::getComponents(std::vector<QueueItemEntity> queueItems, int offset, int countQueueItems)
{
    std::string offsetString = std::to_string(offset);

    dpp::component first = dpp::component().set_label("<<").set_style(dpp::cos_primary).set_id("first" + offsetString);
    dpp::component previous = dpp::component().set_label("<").set_style(dpp::cos_primary).set_id("previous" + offsetString);
    dpp::component current = dpp::component().set_label("[]").set_style(dpp::cos_primary).set_id("current" + offsetString);
    dpp::component next = dpp::component().set_label(">").set_style(dpp::cos_primary).set_id("next" + offsetString);
    dpp::component last = dpp::component().set_label(">>").set_style(dpp::cos_primary).set_id("last" + offsetString);

    if (offset == 0)
    {
        //Already on first page
        first.set_disabled(true);
        previous.set_disabled(true);
    }
    else if (offset > (countQueueItems - 10))
    {
        //Last page
        last.set_disabled(true);
        next.set_disabled(true);
    }

    dpp::component components = dpp::component()
                                    .add_component(first)
                                    .add_component(previous)
                                    .add_component(current)
                                    .add_component(next)
                                    .add_component(last);

    return components;
}

std::string QueueCommand::intToTime(int duration)
{
    std::string infinity = "∞";
    if (duration == -1)
    {
        return infinity;
    }

    int h = std::floor(duration / 3600);
    duration = duration % 3600;
    int m = floor(duration / 60);
    int s = duration % 60;

    char buffer[9];
    if (h > 0)
    {
        std::sprintf(buffer, "%02d:%02d:%02d", h, m, s);
    }
    else
    {
        std::sprintf(buffer, "%02d:%02d", m, s);
    }

    return std::string(buffer);
}

bool QueueCommand::replace(std::string &str, const std::string &from, const std::string &to)
{
    size_t start_pos = str.find(from);
    if (start_pos == std::string::npos)
        return false;
    str.replace(start_pos, from.length(), to);
    return true;
}
