#include "lyrics.h"
#include "../entities/queue-item.h"
#include "../entities/queue.h"
#include "../utils/config.h"
#include "../utils/db.h"
#include "cpr/api.h"
#include "cpr/cprtypes.h"
#include "cpr/curl_container.h"
#include "cpr/response.h"
#include "cpr/timeout.h"
#include <algorithm>
#include <map>
#include <utility>

std::vector<std::string> LyricsCommand::getRegex()
{
    std::vector<std::string> regex;
    regex.push_back("^-lyrics$");
    regex.push_back("^-lürix$");
    return regex;
}

std::string LyricsCommand::getHelp()
{
    return "**-lyrics, -lürix:** Shows the lyrics for the currently playing song";
}

void LyricsCommand::execute(dpp::message_create_t event, std::string match1, std::string match2)
{
    //Get current queue item
    QueueEntity queue = DbUtil::getQueue(event.msg.guild_id);
    if (queue.currentItemId == -1)
    {
        //No current item
        error(event, "No current queue item found");
        return;
    }

    QueueItemEntity currentQueueItem = DbUtil::getCurrentQueueItem(queue.currentItemId);
    if (currentQueueItem.id == -1)
    {
        error(event, "No current queue item found");
        return;
    }

    const std::string lyricsNotFoundError = "Lyrics for current song not found";

    //Get Genius access token
    std::string accessToken = ConfigUtil::get("genius_access_token");
    if (accessToken.empty() || accessToken == "GENIUS_ACCESS_TOKEN")
    {
        error(event, "Please set a Genius access token in the config file");
        return;
    }

    //Search for song on Genius and get its url
    std::string songUrl;
    cpr::Response searchResponse = cpr::Get(cpr::Url("https://api.genius.com/search"),
                                            cpr::Parameters{{"q", currentQueueItem.displayName}},
                                            cpr::Bearer(accessToken),
                                            cpr::Timeout(5000));
    try
    {
        nlohmann::json parsedSearchResponse = nlohmann::json::parse(searchResponse.text);
        int status = parsedSearchResponse.at("meta").at("status");
        if (status != 200)
        {
            error(event, lyricsNotFoundError);
            return;
        }

        nlohmann::json hits = parsedSearchResponse.at("response").at("hits");
        for (nlohmann::json &hit : hits)
        {
            if (hit.at("type") == "song")
            {
                songUrl = hit.at("result").at("url");
                break;
            }
        }
    }
    catch (nlohmann::json::exception e)
    {
        error(event, lyricsNotFoundError);
        return;
    }

    if (songUrl.empty())
    {
        error(event, lyricsNotFoundError);
        return;
    }

    //Get lyrics for song (parse Genius HTML)
    cpr::Response lyricsResponse = cpr::Get(cpr::Url(songUrl), cpr::Timeout(10000));
    std::smatch matches;
    if (std::regex_search(lyricsResponse.text, matches, std::regex("<div data-lyrics-container=\"true\"(.*?)>(.*?)<div class=\"LyricsFooter__Container")))
    {
        std::string lyrics = htmlUnescape(matches[2]); //Unescape HTML entities (e.g. &lt; aka. <, &#x27; aka. ')
        replaceAll(lyrics, "<br/>", "\n");

        //Replace some HTML tags which are in our way
        lyrics = std::regex_replace(lyrics, std::regex("<a(.*?)>"), "");
        lyrics = std::regex_replace(lyrics, std::regex("<span(.*?)>"), "");
        lyrics = std::regex_replace(lyrics, std::regex("<div(.*?)>(.*?)<\\/div>"), "");
        lyrics = std::regex_replace(lyrics, std::regex("<article(.*?)>(.*?)<\\/article>"), "");
        lyrics = std::regex_replace(lyrics, std::regex("<aside(.*?)>(.*?)<\\/aside>"), "");

        //Remove any dangling closing tags
        replaceAll(lyrics, "</a>", "");
        replaceAll(lyrics, "</span>", "");
        replaceAll(lyrics, "</div>", "");
        replaceAll(lyrics, "</article>", "");
        replaceAll(lyrics, "</aside>", "");
        replaceAll(lyrics, "<i>", "");
        replaceAll(lyrics, "</i>", "");
        replaceAll(lyrics, "<b>", "");
        replaceAll(lyrics, "</b>", "");

        //Sometimes there's another data-lyrics-container in the middle of the lyrics
        lyrics = std::regex_replace(lyrics, std::regex("<div data-lyrics-container=\"true\"(.*?)>"), "\n");

        success(event, "Lyrics for \"" + currentQueueItem.displayName + "\"", lyrics);
    }
    else
    {
        error(event, lyricsNotFoundError);
    }
}

void LyricsCommand::replaceAll(std::string &str, const std::string &from, const std::string &to)
{
    if (from.empty())
    {
        return;
    }

    size_t start_pos = 0;
    while ((start_pos = str.find(from, start_pos)) != std::string::npos)
    {
        str.replace(start_pos, from.length(), to);
        start_pos += to.length(); // In case 'to' contains 'from', like replacing 'x' with 'yx'
    }
}

std::string LyricsCommand::htmlUnescape(std::string html)
{
    std::string result = "";
    const char *src;

    src = html.c_str();

    while (*src != '\0')
    {
        if (*src == '&')
        {
            const char *end;
            end = strchr(src, ';');
            src++;

            if (!end)
            {
                //Wrongly encoded
                return html;
            }

            if (src[0] == '#')
            {
                int id, n;

                src++;

                if (src[0] == 'x')
                {
                    n = sscanf(src + 1, "%x", &id);
                }
                else
                {
                    n = sscanf(src, "%u", &id);
                }

                if (n != 1)
                {
                    return html;
                }

                result += id;
            }
            else
            {
                std::map<std::string, std::string> toReplace = {
                    {"amp", "&"},
                    {"lt", "<"},
                    {"gt", ">"},
                    {"nbsp", " "},
                    {"copy", "©"},
                    {"quot", "\""},
                    {"reg", "®"},
                    {"apos", "'"},
                };

                for (const std::pair<std::string, std::string> &ref : toReplace)
                {
                    int len = std::min(ref.first.length(), (size_t)(end - src));
                    if (strncmp(src, ref.first.c_str(), len) == 0)
                    {
                        result += ref.second;
                        break;
                    }
                }
            }
            src = end + 1;
        }
        else
        {
            result += *src;
            src++;
        }
    }

    return result;
}